package com.example.sefinsa_app;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.ExistingWorkPolicy;
import androidx.work.ListenableWorker;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.toolbox.JsonObjectRequest;
import com.example.sefinsa_app.api.API;
import com.example.sefinsa_app.migrations.DatabaseHelper;
import com.example.sefinsa_app.models.Prestamo;
import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;


public class PrestamosWorker extends Worker {
    private SharedPreferences sesion;
    private final ConnectivityManager connectivityManager;
    public static final int BATCH_SIZE_PRESTAMOS = 999999;
    public static int currentPagePrestamos = 0;
    public static boolean allDataLoadedPrestamos = false;
    public static List<Prestamo> allPrestamos = new ArrayList<>();
    public static boolean isTaskPrestamosCompleted = false;

    public PrestamosWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);
        connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
    }

    @RequiresApi(api = Build.VERSION_CODES.N)
    @NonNull
    @Override
    public ListenableWorker.Result doWork() {
        SQLiteDatabase db = null; // Declarar fuera para asegurarte de cerrar después
        try {
            if (!isNetworkAvailable(getApplicationContext())) {
                Log.d("PrestamosWorker", "No internet connection, retrying later.");
                return ListenableWorker.Result.failure(); // Si ocurre un error, marcar como fallo
            }

            if (!isWithinAllowedTime()) {
                Log.d("PrestamosWorker", "Current time is outside allowed range, retrying later.");
                return ListenableWorker.Result.failure(); // Si ocurre un error, marcar como fallo
            }

            // Abrir la base de datos al inicio
            DatabaseHelper dbHelper = new DatabaseHelper(getApplicationContext());
            db = dbHelper.getWritableDatabase();

            // Realizar las operaciones necesarias
            if (!allDataLoadedPrestamos) {
                loadAllData(db); // Pasar la conexión como argumento
                Log.d("PrestamosWorker", "ENTRO EN CARGAR TODOS LOS DATOS Prestamos........................");
            } else {
                checkForUpdates(getApplicationContext(), db); // Pasar la conexión como argumento
                Log.d("ClientesWorker", "ENTRO EN ACTUALIZAR TODOS LOS DATOS Prestamos.......................");
            }
            reprogramarWorker();
            return ListenableWorker.Result.success();

        } catch (Exception e) {
            Log.e("PrestamosWorker", "Error in doWork", e);
            return ListenableWorker.Result.failure();
        } finally {
            if (isTaskPrestamosCompleted && db != null && db.isOpen()) {
                db.close();
            }
        }

    }

    private void reprogramarWorker() {
        WorkManager.getInstance(getApplicationContext()).enqueueUniqueWork(
                "PrestamosWorker",
                ExistingWorkPolicy.REPLACE, // Evita que se creen trabajos duplicados
                new OneTimeWorkRequest.Builder(PrestamosWorker.class)
                        .setInitialDelay(10, TimeUnit.MINUTES)
                        .build()
        );

        Log.d("AvalesWorker", "Worker PrestamosWorker reprogramado para ejecutarse en 9 minuto.");
    }

    private void loadAllData(SQLiteDatabase db) {
        Handler mainHandler = new Handler(Looper.getMainLooper());
        Context context = getApplicationContext();

        if (context == null) {
            Log.e("PrestamosWorker", "Contexto es nulo. No se puede cargar la data.");
            return;
        }

        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);

        JSONObject data = new JSONObject();
        try {
            data.put("page", currentPagePrestamos);
            data.put("size", BATCH_SIZE_PRESTAMOS);
            String nombrePerfil = sesion.getString("nombre_perfil", "");

            switch (nombrePerfil) {
                case "COBRADOR":
                case "GESTOR":
                    data.put("func", "prestamosConSemanasSinPagar");
                    break;

                case "JURIDICO":
                    data.put("func", "prestamosConSemanasSinPagar");
                    data.put("estatus", "4");
                    break;

                default:
                    data.put("func", "index_app_lotes");
                    data.put("estatus", "0");
                    break;
            }

            Log.d("PRESTAMOS", "Parámetros enviados WORKER Prestamos: " + data);
        } catch (JSONException e) {
            Log.e("PrestamosWorker", "Error construyendo JSON", e);
            return;
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPrestamos, data,
                response -> {
                        try {
                            JSONArray data1 = response.getJSONArray("data");
                            List<Prestamo> prestamosBatch = new ArrayList<>();

                            for (int i = 0; i < data1.length(); i++) {
                                JSONObject obj = data1.getJSONObject(i);
                                Gson gson = new Gson();
                                Prestamo prestamo = gson.fromJson(obj.toString(), Prestamo.class);
                                prestamosBatch.add(prestamo);
                            }
                            for (Prestamo prestamo : prestamosBatch) {
                                Log.d("PRESTAMOS", "📄 Préstamo a insertar: Tarjetón: " + prestamo.getNumero_tarjeton() +
                                        ", Updated_at: " + prestamo.getUpdated_at() +
                                        ", Id Prestamo: " + prestamo.getId() +
                                        ", Nombre_completo: " + prestamo.getNombre_completo());
                            }
                            // Inserción en lote
                            insertPrestamoIntoSQLiteAllData(context, prestamosBatch, db);

                            // Verificar si se cargaron todos los datos
                            if (prestamosBatch.size() < BATCH_SIZE_PRESTAMOS) {
                                allDataLoadedPrestamos = true;
                            }

                        } catch (JSONException e) {
                            Log.e("PrestamosWorker", "Error procesando respuesta JSON", e);
                        }
                },
                error -> mainHandler.post(() -> handleNetworkError(context))
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                30000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void insertPrestamoIntoSQLiteAllData(Context context, List<Prestamo> prestamos, SQLiteDatabase db) {
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }
        try {
            for (Prestamo prestamo : prestamos) {
                ContentValues values = new ContentValues();
                values.put("id", prestamo.getId());
                values.put("cliente_id", prestamo.getCliente_id());
                values.put("direccion_cliente", prestamo.getDireccion_cliente());
                values.put("telefono_cliente", prestamo.getTelefono_cliente());
                values.put("ruta_id", prestamo.getRuta_id());
                values.put("poblacion_id", prestamo.getPoblacion_id());
                values.put("colocadora_id", prestamo.getColocadora_id());
                values.put("aval_id", prestamo.getAval_id());
                values.put("grupo_poblacion", prestamo.getGrupo_poblacion() != null ? prestamo.getGrupo_poblacion() : "NULLOO");
                values.put("monto_prestado", prestamo.getMonto_prestado() != null ? prestamo.getMonto_prestado() : "NULLOO");
                values.put("pago_semanal", prestamo.getPago_semanal() != null ? prestamo.getPago_semanal() : "NULLOO");
                values.put("fecha_prestamo", prestamo.getFecha_prestamo());
                values.put("modalidad_semanas", prestamo.getModalidad_semanas());
                values.put("numero_tarjeton", prestamo.getNumero_tarjeton());
                values.put("status", prestamo.getStatus());
                values.put("created_at", prestamo.getCreated_at());
                values.put("updated_at", prestamo.getUpdated_at());

                // 📌 Verificar si ya existe el préstamo en la base de datos
                Cursor cursor = db.rawQuery("SELECT id FROM prestamos WHERE id = ?", new String[]{String.valueOf(prestamo.getId())});
                //boolean exists = cursor.moveToFirst();
                cursor.close();

                long newRowId = db.insertWithOnConflict("prestamos", null, values, SQLiteDatabase.CONFLICT_IGNORE);
                if (newRowId == -1) {
                    db.update("prestamos", values, "id = ?", new String[]{prestamo.getId()});
                }
                else
                {
                    Log.d("SQLite", "✅ Préstamo insertado en BD: Tarjetón: " + prestamo.getNumero_tarjeton() +
                            ", Id: " + prestamo.getId() + ", Nombre: " + prestamo.getNombre_completo());
                }
            }
        } catch (Exception e) {
            Log.e("SQLite", "Error durante la inserción en lote", e);
        } finally {
            isTaskPrestamosCompleted = true;
        }
    }


    private void handleNetworkError(Context context) {
        if (isNetworkAvailable(context)) {
            //Log.d("PrestamosWorker", "Network available, resuming work.");
            enqueueWork(context); // Siempre intenta encolar el trabajo si hay red disponible
        } else {
            //Log.d("PrestamosWorker", "No network available, stopping work.");
            cancelAndRescheduleWork(context);
        }
    }
    private void cancelAndRescheduleWork(Context context) {
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();

        OneTimeWorkRequest retryWorkRequest = new OneTimeWorkRequest.Builder(PrestamosWorker.class)
                .setConstraints(constraints)
                .setInitialDelay(10, TimeUnit.MINUTES) // Reprogramar después de 5 minutos
                .build();

        WorkManager.getInstance(context).enqueue(retryWorkRequest);
        Log.d("PrestamosWorker", "Reprogramado trabajo debido a error.");

        //Log.d("PrestamosWorker", "Scheduled periodic work to check network availability.");
    }
    private boolean isWithinAllowedTime() {
        Calendar calendar = Calendar.getInstance();
        int hourOfDay = calendar.get(Calendar.HOUR_OF_DAY);
        return hourOfDay >= 7 && hourOfDay <= 22;
    }

    private boolean isNetworkAvailable(Context context) {
        try {
            NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
            return activeNetworkInfo != null && activeNetworkInfo.isConnected();
        } catch (Exception e) {
            Log.e("PrestamosWorker", "Error checking network availability", e);
            return false;
        }
    }

    public static void enqueueWork(Context context) {
        // Configurar restricciones para el trabajo
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED) // Requiere conexión a Internet
                .build();

        // Crear un OneTimeWorkRequest para que el trabajo se ejecute una sola vez
        OneTimeWorkRequest oneTimeWorkRequest = new OneTimeWorkRequest.Builder(PrestamosWorker.class)
                .setConstraints(constraints) // Aplicar las restricciones
                //.setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 10, TimeUnit.MINUTES) // Configurar reintentos en caso de fallo
                .build();

        // Encolar el trabajo único
        WorkManager.getInstance(context.getApplicationContext()).enqueueUniqueWork(
                "PrestamosWorker", // Identificador único para el trabajo
                ExistingWorkPolicy.REPLACE, // Reemplaza el trabajo si ya existe uno en cola con el mismo nombre
                oneTimeWorkRequest
        );

        Log.d("PrestamosWorker", "Se ha encolado un OneTimeWorkRequest para PrestamosWorker.");
    }


    private void checkForUpdates(Context context, SQLiteDatabase db) {
        if (context == null) {
            Log.e("PrestamosWorker", "Context is null, cannot proceed with checkForUpdates.");
            return;
        }

        // Obtener préstamos pendientes con status = 8
        List<Map<String, String>> prestamosPendientes = obtenerPrestamosPendientes(context, db);
        if (!prestamosPendientes.isEmpty()) {
            Log.d("PrestamosWorker", "Encontrados " + prestamosPendientes.size() + " préstamos pendientes para enviar al servidor.");

            // Enviar cada préstamo pendiente al servidor
            for (Map<String, String> prestamo : prestamosPendientes) {
                insertDatosLocalAlServidor(context, prestamo, db);
            }
        } else {
            Log.d("PrestamosWorker", "No hay préstamos pendientes de enviar al servidor.");
        }

        Gson gson = new Gson();
        DatabaseHelper dbHelper = new DatabaseHelper(context);
        List<Prestamo> localPrestamos = dbHelper.getLast40PrestamosFromSQLite();
        // Construcción del objeto JSON para la solicitud
        JSONObject data = new JSONObject();
        try {
            String nombrePerfil = sesion.getString("nombre_perfil", "");

            switch (nombrePerfil) {
                case "COBRADOR":
                case "GESTOR":
                    data.put("func", "prestamosConSemanasSinPagar");
                    break;

                case "JURIDICO":
                    data.put("func", "prestamosConSemanasSinPagar");
                    data.put("estatus", "4");
                    break;

                default:
                    data.put("func", "indexApp_Upd");
                    data.put("estatus", "0");
                    break;
            }
            //Log.d("PRESTAMOS", "Parametros enviados WORKER Prestamos: " + data.toString());
        } catch (JSONException e) {
            e.printStackTrace();
            return; // En caso de error, salir del método
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPrestamos, data,
                response -> {
                        try {
                            JSONArray data1 = response.getJSONArray("data");
                            List<Prestamo> newPrestamos = new ArrayList<>();

                            // Procesar cada objeto JSON en la respuesta
                            for (int i = 0; i < data1.length(); i++) {
                                JSONObject obj = data1.getJSONObject(i);
                                Prestamo prestamo = gson.fromJson(obj.toString(), Prestamo.class);
                                newPrestamos.add(prestamo);
                            }
                            updatePrestamoInSQLite(context, newPrestamos,localPrestamos, db);
                        } catch (JSONException e) {
                            e.printStackTrace();
                        }
                }, error -> {
            Log.e("PrestamosWorker", "Error en la solicitud de actualización de datos de prestamos: " + error.toString());
        }
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                30000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void updatePrestamoInSQLite(Context context, List<Prestamo> serverData, List<Prestamo> localPrestamos, SQLiteDatabase db) {
        //Log.d("updatePrestamoInSQLite", "📌 Iniciando actualización de préstamos. Cantidad en servidor: " + serverData.size() + ", cantidad en local: " + localPrestamos.size());
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }
        // Mostrar los números de tarjetón de serverData en el log
        for (Prestamo prestamo : serverData) {
            //Log.d("updatePrestamoInSQLite", "📄 Préstamo en servidor - Número Tarjetón: " + prestamo.getNumero_tarjeton());
        }

        // Crear un conjunto para buscar rápidamente si un ID existe en la lista local
        Set<String> localPrestamoIds = new HashSet<>();
        for (Prestamo localPrestamo : localPrestamos) {
            localPrestamoIds.add(localPrestamo.getNumero_tarjeton());
        }

        // Inicializar contador
        AtomicInteger remainingComparisons = new AtomicInteger(serverData.size());

        // Sincronizar clientes desde el servidor a la base de datos local
        for (Prestamo serverPrestamo : serverData) {
            //Log.d("updatePrestamoInSQLite", "🔍 Analizando préstamo con tarjetón: " + serverPrestamo.getNumero_tarjeton());

            if (localPrestamoIds.contains(serverPrestamo.getNumero_tarjeton())) {
                //Log.d("updatePrestamoInSQLite", "✅ Préstamo encontrado en la base de datos local.");

                // Buscar el préstamo local correspondiente
                Prestamo localPrestamo = null;
                for (Prestamo prestamo : localPrestamos) {
                    if (prestamo.getNumero_tarjeton().equals(serverPrestamo.getNumero_tarjeton())) {
                        localPrestamo = prestamo;
                        break;
                    }
                }

                if (localPrestamo != null) {
                    String localUpdatedAt = localPrestamo.getUpdated_at();
                    String serverUpdatedAt = serverPrestamo.getUpdated_at();

                    if (localUpdatedAt == null || localUpdatedAt.isEmpty()) {
                        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
                            LocalDateTime now = LocalDateTime.now(); // Hora actual
                            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                            localUpdatedAt = now.format(formatter);
                        } else {
                            // Manejo para versiones antiguas (opcional, pero recomendable si soportas < API 26)
                            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                            localUpdatedAt = sdf.format(new Date()); // Hora actual sin modificaciones
                        }
                        // Insertar la hora actual en la base de datos SQLite
                        ContentValues values = new ContentValues();
                        values.put("updated_at", localUpdatedAt);

                        int rowsUpdated = db.update("prestamos", values, "id = ?", new String[]{localPrestamo.getId()});

                        if (rowsUpdated > 0) {
                            Log.d("SQLite", "Updated 'updated_at' for prestamos ID: " + localPrestamo.getId());
                        } else {
                            Log.e("SQLite", "Failed to update 'updated_at' for prestamos ID: " + localPrestamo.getId());
                        }
                    }

                    //Log.d("updatePrestamoInSQLite", "📅 Fechas comparadas -> Local: " + localUpdatedAt + ", Servidor: " + serverUpdatedAt);

                    if (localUpdatedAt.isEmpty() || isNewer(serverUpdatedAt, localUpdatedAt)) {
                        //Log.d("updatePrestamoInSQLite", "🔄 El servidor tiene una versión más nueva. Actualizando en SQLite.");
                        updateValuesInDatabase(db, serverPrestamo, remainingComparisons);

                    } else if (isNewer(localUpdatedAt, serverUpdatedAt)) {
                        //Log.d("updatePrestamoInSQLite", "⬆️ La versión local es más reciente. Enviando datos al servidor.");
                        enviarDatosLocalAlServidor(context, localPrestamo, db, remainingComparisons);

                    } else {
                        //Log.d("updatePrestamoInSQLite", "✅ Los datos son iguales. No se requiere actualización.");
                        remainingComparisons.decrementAndGet();
                    }
                }
            } else {
                //Log.d("updatePrestamoInSQLite", "🆕 Préstamo no encontrado en local. Insertando en SQLite.");
                insertPrestamoIntoSQLite(context, Collections.singletonList(serverPrestamo), db, remainingComparisons);
            }
        }

        // Verificar si es hora de cerrar la base de datos
        if (remainingComparisons.get() == 0) {
            //Log.d("updatePrestamoInSQLite", "📌 No quedan comparaciones pendientes. Cerrando conexión a la base de datos.");
            isTaskPrestamosCompleted = true;
        }
    }


    private void insertPrestamoIntoSQLite(Context context, List<Prestamo> prestamos, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        try {
            for (Prestamo prestamo : prestamos) {
                ContentValues values = new ContentValues();
                values.put("id", prestamo.getId());
                values.put("cliente_id", prestamo.getCliente_id());
                values.put("direccion_cliente", prestamo.getDireccion_cliente());
                values.put("telefono_cliente", prestamo.getTelefono_cliente());
                values.put("ruta_id", prestamo.getRuta_id());
                values.put("poblacion_id", prestamo.getPoblacion_id());
                values.put("colocadora_id", prestamo.getColocadora_id());
                values.put("aval_id", prestamo.getAval_id());
                values.put("grupo_poblacion", prestamo.getGrupo_poblacion() != null ? prestamo.getGrupo_poblacion() : "NULLOO");
                values.put("monto_prestado", prestamo.getMonto_prestado() != null ? prestamo.getMonto_prestado() : "NULLOO");
                values.put("pago_semanal", prestamo.getPago_semanal() != null ? prestamo.getPago_semanal() : "NULLOO");
                values.put("fecha_prestamo", prestamo.getFecha_prestamo());
                values.put("modalidad_semanas", prestamo.getModalidad_semanas());
                values.put("numero_tarjeton", prestamo.getNumero_tarjeton());
                values.put("status", prestamo.getStatus());
                values.put("created_at", prestamo.getCreated_at());
                values.put("updated_at", prestamo.getUpdated_at());

                long newRowId = db.insertWithOnConflict("prestamos", null, values, SQLiteDatabase.CONFLICT_IGNORE);
                remainingComparisons.decrementAndGet();
                if (newRowId == -1) {
                    db.update("prestamos", values, "id = ?", new String[]{prestamo.getId()});
                }
            }
        } catch (Exception e) {
            remainingComparisons.decrementAndGet();
            Log.e("SQLite", "Error durante la inserción en lote", e);
        }
    }

    private void enviarDatosLocalAlServidor(Context context, Prestamo prestamoLocal, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        Log.d("PrestamoWorker", "Sending local data to the server PRESTAMOS...");

        Handler mainHandler = new Handler(Looper.getMainLooper());

        if (context == null) {
            Log.e("PrestamosWorker", "Context is null. Cannot send data to server.");
            return; // Salir si el contexto es nulo
        }

        JSONObject data = new JSONObject();
        try {
            data.put("func", "updateFromLocal");
            data.put("id", prestamoLocal.getId());
            data.put("aval_id", prestamoLocal.getAval_id());
            data.put("updated_at", prestamoLocal.getUpdated_at());// Agregar el campo updated_at

            Log.d("PRESTSMOS", "Parametros PRESTAMOS enviados al servidor para actualizar: " + data);
        } catch (JSONException e) {
            e.printStackTrace();
            return;  // Si hay un error en la creación del JSON, salir
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPrestamos, data,
                response -> {
                    remainingComparisons.decrementAndGet();
                    Log.d("PrestamosWorker", "Datos locales enviados y respuesta recibida PRESTAMOS: " + response.toString());
                },
                error -> {
                    mainHandler.post(() -> {
                        String errorMessage = error.getMessage();
                        Log.e("PrestamosWorker", "ERROR occurred al enviar datos locales Prestamos: " + errorMessage, error);
                        remainingComparisons.decrementAndGet();
                        // Comprobar el tipo de error (puede ser un error de red, tiempo de espera, etc.)
                        if (error.networkResponse != null) {
                            Log.e("PrestamosWorker", "ERROR Código de estado PRESTAMOS: " + error.networkResponse.statusCode);
                        }
                    });
                }
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                10000, // 10 segundos de timeout
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    @SuppressLint("Range")
    private List<Map<String, String>> obtenerPrestamosPendientes(Context context, SQLiteDatabase db) {
        List<Map<String, String>> prestamosPendientes = new ArrayList<>();

        String query = "SELECT * FROM prestamos WHERE status = 8";
        Cursor cursor = db.rawQuery(query, null);

        if (cursor != null && cursor.moveToFirst()) {
            do {
                Map<String, String> prestamo = new HashMap<>();
                prestamo.put("id", cursor.getString(cursor.getColumnIndex("id")));
                prestamo.put("cliente_id", cursor.getString(cursor.getColumnIndex("cliente_id")));
                prestamo.put("aval_id", cursor.getString(cursor.getColumnIndex("aval_id")));
                prestamo.put("monto_prestado", cursor.getString(cursor.getColumnIndex("monto_prestado")));
                prestamo.put("pago_semanal", cursor.getString(cursor.getColumnIndex("pago_semanal")));
                prestamo.put("fecha_prestamo", cursor.getString(cursor.getColumnIndex("fecha_prestamo")));
                prestamo.put("modalidad_semanas", cursor.getString(cursor.getColumnIndex("modalidad_semanas")));
                prestamo.put("numero_tarjeton", cursor.getString(cursor.getColumnIndex("numero_tarjeton")));
                prestamo.put("grupo_poblacion", cursor.getString(cursor.getColumnIndex("grupo_poblacion")));
                prestamo.put("status", cursor.getString(cursor.getColumnIndex("status")));

                prestamosPendientes.add(prestamo);
            } while (cursor.moveToNext());
        }

        return prestamosPendientes;
    }

    private void insertDatosLocalAlServidor(final Context context, final Map<String, String> prestamoLocal, SQLiteDatabase db) {
        Log.d("PrestamoWorker", "Sending insertDatosLocalAlServidor local data to the server PRESTAMOS...");

        Handler mainHandler = new Handler(Looper.getMainLooper());
        if (context == null) {
            Log.e("PrestamosWorker", "Context is null. Cannot send data to server.");
            return; // Salir si el contexto es nulo
        }

        JSONObject data = new JSONObject();
        try {
            // Obtén los valores del mapa en lugar de usar métodos de la clase `Prestamo`
            data.put("func", "create");
            data.put("cliente_id", prestamoLocal.get("cliente_id"));
            data.put("aval_id", prestamoLocal.get("aval_id"));
            data.put("grupo", prestamoLocal.get("grupo_poblacion"));
            data.put("monto_prestado", prestamoLocal.get("monto_prestado"));
            data.put("pago_semanal", prestamoLocal.get("pago_semanal"));
            data.put("fecha_prestamo", prestamoLocal.get("fecha_prestamo"));
            data.put("modalidad", prestamoLocal.get("modalidad_semanas"));
            data.put("tarjeton", prestamoLocal.get("numero_tarjeton"));

            Log.d("PrestamoWorker", "Parametros enviados al servidor: " + data);
        } catch (JSONException e) {
            Log.e("PrestamoWorker", "Error al crear JSON para el préstamo: " + e.getMessage(), e);
            return; // Si hay un error en la creación del JSON, salir
        }

        // Realizar la solicitud al servidor
        Log.d("PrestamoWorker", "Realizando la solicitud POST al servidor: " + API.urlPrestamos);
        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPrestamos, data,
                response -> {
                    Log.d("PrestamoWorker", "Datos locales enviados y respuesta recibida PRESTAMOS: " + response.toString());

                    try {
                        // Acceder al objeto "data" en la respuesta
                        JSONObject data2 = response.getJSONObject("data");

                        // Ahora obtener el "prestamo_id" desde el objeto "data"
                        String prestamoIdServidor = data2.getString("prestamo_id");

                        Log.d("PrestamoWorker", "ID del préstamo recibido del servidor: " + prestamoIdServidor);

                        // Si la respuesta es exitosa y hay un ID, actualiza el préstamo en SQLite
                        if (prestamoIdServidor != null && !prestamoIdServidor.isEmpty()) {
                            String prestamoIdLocal = prestamoLocal.get("id");
                            Log.d("PrestamoWorker", "Actualizando el préstamo en SQLite con el ID: " + prestamoIdServidor);
                            updatePrestamoIdInSQLite(context, prestamoIdLocal, prestamoIdServidor, db);
                        } else {
                            Log.e("PrestamoWorker", "El servidor no ha devuelto un prestamo_id válido.");
                        }
                    } catch (JSONException e) {
                        Log.e("PrestamoWorker", "Error al procesar la respuesta del servidor: " + e.getMessage(), e);
                    }
                },
                error -> mainHandler.post(() -> {
                    String errorMessage = error.getMessage();
                    Log.e("PrestamoWorker", "ERROR al enviar datos locales Prestamos: " + errorMessage, error);

                    if (error.networkResponse != null) {
                        Log.e("PrestamoWorker", "Código de respuesta del error: " + error.networkResponse.statusCode);
                    }
                })
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                10000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void updatePrestamoIdInSQLite(Context context, String prestamoIdServidor, String idServidor, SQLiteDatabase db) {
            // Crear un ContentValues para actualizar el préstamo local con el `id` asignado por el servidor
            ContentValues values = new ContentValues();
            values.put("id", idServidor);  // Actualiza el `prestamo_id` con el ID del servidor
            values.put("status", 0);

            // Actualizar el préstamo en SQLite con el `id` del servidor
            int rowsUpdatedPrestamos = db.update("prestamos", values, "id = ?", new String[]{prestamoIdServidor});

            if (rowsUpdatedPrestamos > 0) {
                Log.d("SQLite", "Prestamo actualizado correctamente con el ID del servidor: " + idServidor);
            } else {
                Log.e("SQLite", "No se pudo actualizar el préstamo en SQLite.");
            }

            // Ahora actualizamos la tabla de pagos, donde el prestamo_id coincida con el prestamoIdServidor
            ContentValues paymentValues = new ContentValues();
            paymentValues.put("prestamo_id", idServidor);  // Actualiza el prestamo_id en la tabla pagos

            int rowsUpdatedPagos = db.update("pagos", paymentValues, "prestamo_id = ?", new String[]{prestamoIdServidor});

            if (rowsUpdatedPagos > 0) {
                Log.d("SQLite", "Pagos actualizados correctamente con el nuevo prestamo_id: " + idServidor);
            } else {
                Log.e("SQLite", "No se pudo actualizar los pagos en SQLite.");
            }
    }

    private void updateValuesInDatabase(SQLiteDatabase db, Prestamo prestamo, AtomicInteger remainingComparisons) {
            ContentValues values = new ContentValues();
            values.put("id", prestamo.getId());  // Suponiendo que tienes un prestamo local
            values.put("cliente_id", prestamo.getCliente_id());
            values.put("direccion_cliente", prestamo.getDireccion_cliente());
            values.put("telefono_cliente", prestamo.getTelefono_cliente());
            values.put("ruta_id", prestamo.getRuta_id());
            values.put("poblacion_id", prestamo.getPoblacion_id());
            values.put("colocadora_id", prestamo.getColocadora_id());
            values.put("aval_id", prestamo.getAval_id());
            values.put("grupo_poblacion", prestamo.getGrupo_poblacion());
            values.put("monto_prestado", prestamo.getMonto_prestado());
            values.put("pago_semanal", prestamo.getPago_semanal());
            values.put("fecha_prestamo", prestamo.getFecha_prestamo());// Agregar el campo updated_at
            values.put("modalidad_semanas", prestamo.getModalidad_semanas());// Agregar el campo updated_at
            values.put("numero_tarjeton", prestamo.getNumero_tarjeton());// Agregar el campo updated_at
            values.put("status", prestamo.getStatus());// Agregar el campo updated_at
            values.put("created_at", prestamo.getCreated_at());// Agregar el campo updated_at
            values.put("updated_at", prestamo.getUpdated_at());// Agregar el campo updated_at

            // Actualizar el registro en la base de datos
            int rowsUpdated = db.update("prestamos", values, "id=?", new String[]{String.valueOf(prestamo.getId())});
            remainingComparisons.decrementAndGet();
            if (rowsUpdated > 0) {
                Log.d("SQLite", "Registro actualizado correctamente prestamos: ID = " + prestamo.getId());
            } else {
                Log.e("SQLite", "Error al actualizar el registro. ID no encontrado o fallo en la base de datos.");
            }
    }

    private boolean isNewer(String serverDate, String localDate) {
        // Verificar si alguna de las fechas es nula o vacía
        if (serverDate == null || localDate == null || serverDate.isEmpty() || localDate.isEmpty()) {
            Log.e("isNewer", "Una de las fechas es nula o vacía: serverDate=" + serverDate + ", localDate=" + localDate);
            return false; // O decide cómo manejar este caso
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX")); // Ajusta el formato según tu necesidad
        try {
            Date server = sdf.parse(serverDate);
            Date local = sdf.parse(localDate);

            return server != null && local != null && server.after(local);
        } catch (ParseException e) {
            e.printStackTrace();
            return false; // En caso de error, asumir que no es más reciente
        }
    }
}

